/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/

/**
 * \file    aic3106_lineout_out.c
 *
 * \brief   Routines to initialize the AIC3106 Audio codec for Line-Out output
 *
 * This file contains the routines for configuring the AIC3106 Audio Codec for 
 * driving the sine tone output to the Head-phones.
 *
 * \author  0043
 *
 * \version 01a,18aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "DM814x_McASP.h"
#include "DM814x_i2c.h"
#include "aic3106.h"

/*
 *====================
 * Defines
 *====================
 */
#define 	AIC3106_LO_TEST_DURATION	5
/**<Duration (in seconds) for which the test has to be run */
 
/*
 *====================
 * Static variables
 *====================
 */ 
 /* Pre-generated sine wave data, 16-bit signed samples */
static INT16 sinetable[48] = {
    0x0000, 0x10b4, 0x2120, 0x30fb, 0x3fff, 0x4dea, 0x5a81, 0x658b,
    0x6ed8, 0x763f, 0x7ba1, 0x7ee5, 0x7ffd, 0x7ee5, 0x7ba1, 0x763f,
    0x6ed8, 0x658b, 0x5a81, 0x4dea, 0x3fff, 0x30fb, 0x2120, 0x10b4,
    0x0000, 0xef4c, 0xdee0, 0xcf06, 0xc002, 0xb216, 0xa57f, 0x9a75,
    0x9128, 0x89c1, 0x845f, 0x811b, 0x8002, 0x811b, 0x845f, 0x89c1,
    0x9128, 0x9a76, 0xa57f, 0xb216, 0xc002, 0xcf06, 0xdee0, 0xef4c
};

/*
 *====================
 * Function implementations
 *====================
 */
/** **************************************************************************
 * \n \brief Routine to configure the Headphone Out and play sine tone
 *
 * This routine initializes the AIC3106 Audio Codec on the base board and
 * plays a sine tone on the head-phone.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
STATUS aic3106_lineout_out 
(
	UINT8    u8McaspNum
)
{
	STATUS u32RetVal = SUCCESS;
	DM814x_MCASP_REGS stMcAspRegs = NULL;
	UINT16 u16Sample = 0;
	UINT16 u16MilliSec = 0;
	UINT16 u16Seconds = 0;

	if (u8McaspNum == 2)
		stMcAspRegs = (DM814x_MCASP_REGS) DM814X_EVM_AIC3106_MCASP;
		
    /* Configure AIC3106 */	
    aic3106RegSet(  AIC3106_PAGESELECT, 0 );       // Select page 0
    aic3106RegSet(  AIC3106_RESET, 0x80 );         // Reset AIC3106
    /* ------------------------------------------------------------------------ *
     *                                                                          *
     *  AIC3106 Setup                                                           *
     *                                                                          *
     *      AIC3106.MCLK = PLL1705.SCK02                                        *
     *      FS = ( AIC3106.MCLK * K ) / ( 2048 * P )                            *
     *                                                                          *
     *      For a FS=[48 kHz] & MCLK=[24.576 MHz]                              *
     *          : 48kHz = ( 24.576 MHz * K ) / ( 2048 * P )                    *
     *          : P = 2, K[J.D] = 8.7075                                        *
     *                                                                          *
     * ------------------------------------------------------------------------ */
    /* Configure AIC3106 registers */
    aic3106RegSet(  3, 0x22 );  // 5 PLL A                            <- [PLL=OFF][Q=4][P=2]
    aic3106RegSet(  4, 0x20 );  // 4 PLL B                            <- [J=8]
    aic3106RegSet(  5, 0x6E );  // 5 PLL C                            <- [D=7075]
    aic3106RegSet(  6, 0x23 );  // 6 PLL D                            <- [D=7075]
    aic3106RegSet(  7, 0x0A );  // 7 Codec Datapath Setup             <- [FS=48 kHz][LeftDAC=LEFT][RightDAC=RIGHT]
    aic3106RegSet(  8, 0xC0 );  // 8  Audio Interface Control A       <- [BCLK=Slave][MCLK=Slave]
	aic3106RegSet(  9, 0x00 );  // 9  Audio Interface Control B       <- [I2S mode][16 bit]
	aic3106RegSet(  10, 0x00);  // 10 Audio Interface Control C       <- [Data offset=0]
  	aic3106RegSet(  15, 0 );    // 15  Left ADC PGA Gain              <- [Mute=OFF]
   	aic3106RegSet(  16, 0 );    // 16 Right ADC PGA Gain              <- [Mute=OFF]
   	aic3106RegSet(  19, 0x04 ); // 19  LINE1L to  Left ADC            <- [SingleEnd][Gain=0dB][Power=ON][SoftStep=OncePerFS]
   	aic3106RegSet(  22, 0x04 ); // 22  LINE1R to Right ADC            <- [SingleEnd][Gain=0dB][Power=ON][SoftStep=OncePerFS]
   	aic3106RegSet(  27, 0 );    // 27  Left AGC B                     <- [OFF]
   	aic3106RegSet(  30, 0 );    // 30 Right AGC B                     <- [OFF]
   	aic3106RegSet(  37, 0xE0 ); // 37 DAC Power & Output Dvr          <- [LeftDAC=ON][RightDAC=ON][HPLCOM=SingleEnd]
   	aic3106RegSet(  38, 0x10 ); // 38 High Power Output Dvr           <- [HPRCOM=SingleEnd][ShortCircuit=OFF]
    aic3106RegSet(  43, 0 );    // 43  Left DAC Digital Volume        <- [Mute=OFF][Gain=0dB]
  	aic3106RegSet(  44, 0 );    // 44 Right DAC Digital Volume        <- [Mute=OFF][Gain=0dB]
   	aic3106RegSet(  58, 0 );    // 58           HPLCOM Output         <- []
   	aic3106RegSet(  72, 0 );    // 72           HPRCOM Output         <- []
    aic3106RegSet(  82, 0x80 ); // 82 DAC_L1 to LEFT_LOP/M Volume     <- [Routed]
    aic3106RegSet(  86, 0x09 ); // 86           LEFT_LOP/M Output     <- [Mute=OFF][Power=ON]
    aic3106RegSet(  92, 0x80 ); // 92 DAC_R1 to RIGHT_LOP/M Volume    <- [Routed]
    aic3106RegSet(  93, 0x09 ); // 93           RIGHT_LOP/M Output    <- [Mute=OFF][Power=ON]
    aic3106RegSet( 101, 0x01 ); // 101 GPIO Control Register B        <- [CODEC_CLKIN = CLKDIV_OUT]
	aic3106RegSet( 102, 0 );    // 102 Clock Generation Control       <- [PLLCLK_IN and CLKDIV_IN use MCLK]
	
    /* Initialize MCASP2 */
    /* ---------------------------------------------------------------- *
     *                                                                  *
     *  McASP2 is in MASTER mode.                                       *
     *      BCLK & WCLK come from McASP2                                *
     *      DIN is used by write16/write32                              *
     *      DOUT is usec by read16/read32                               *
     *                                                                  *
     * ---------------------------------------------------------------- */
	stMcAspRegs->u32GBLCTL = 0;		// Reset the McASP
	stMcAspRegs->u32RGBLCTL = 0; 	// Reset the Rx section
	stMcAspRegs->u32XGBLCTL	= 0;	// Reset the Tx section
	
	/* Rx side initialization */
	stMcAspRegs->u32RMASK	= 0xFFFFFFFF;	// No padding used
	stMcAspRegs->u32RFMT	= 0x00018078; // MSB 16bit, 1-delay, no pad, CFGBus
	stMcAspRegs->u32AFSRCTL = 0x00000112; // 2TDM, 1bit Rising, INTERNAL FS, word
	stMcAspRegs->u32ACLKRCTL  = 0x000000AF; // Rising INTERNAL CLK (from tx side)
    stMcAspRegs->u32AHCLKRCTL  = 0x00000000; // INT CLK (from tx side)
    stMcAspRegs->u32RTDM       = 0x00000003; // Slots 0,1
    stMcAspRegs->u32RINTCTL    = 0x00000000; // Not used
    stMcAspRegs->u32RCLKCHK    = 0x00FF0008; // 255-MAX 0-MIN, div-by-256

    /* TX */
    stMcAspRegs->u32XMASK      = 0xffffffff; // No padding used
    stMcAspRegs->u32XFMT       = 0x00018078; // MSB 16bit, 1-delay, no pad, CFGBus
    stMcAspRegs->u32AFSXCTL    = 0x00000110; // 2TDM, 1bit Rising edge INTERNAL FS, word
    stMcAspRegs->u32ACLKXCTL   = 0x0000008F; // ASYNC, Rising INTERNAL CLK, div-by-16
    stMcAspRegs->u32AHCLKXCTL  = 0x00000000; // EXTERNAL CLK, div-by-1
    stMcAspRegs->u32XTDM       = 0x00000003; // Slots 0,1
    stMcAspRegs->u32XINTCTL    = 0x00000000; // Not used
    stMcAspRegs->u32XCLKCHK    = 0x00FF0008; // 255-MAX 0-MIN, div-by-256

    stMcAspRegs->u32SRCTL0     = 0x000D;     // MCASP2.AXR0 --> DIN
    stMcAspRegs->u32SRCTL1     = 0x000E;     // MCASP2.AXR1 <-- DOUT
    stMcAspRegs->u32PINFUNC      = 0;          // All MCASPs
    stMcAspRegs->u32PDIR       = 0x00000001; // All inputs except AXR0, ACLKX1, AFSX1

    stMcAspRegs->u32DITCTL     = 0x00000000; // Not used
    stMcAspRegs->u32DLBCTL     = 0x00000000; // Not used
    stMcAspRegs->u32AMUTE      = 0x00000000; // Not used


    /* Starting sections of the McASP*/
    stMcAspRegs->u32XGBLCTL |= GBLCTL_XHCLKRST_ON;                                    // HS Clk
    while ( ( stMcAspRegs->u32XGBLCTL & GBLCTL_XHCLKRST_ON ) != GBLCTL_XHCLKRST_ON );  
    stMcAspRegs->u32RGBLCTL |= GBLCTL_RHCLKRST_ON;                                    // HS Clk
    while ( ( stMcAspRegs->u32RGBLCTL & GBLCTL_RHCLKRST_ON ) != GBLCTL_RHCLKRST_ON );
   
    stMcAspRegs->u32XGBLCTL |= GBLCTL_XCLKRST_ON;                                     // Clk
    while ( ( stMcAspRegs->u32XGBLCTL & GBLCTL_XCLKRST_ON ) != GBLCTL_XCLKRST_ON );
    stMcAspRegs->u32RGBLCTL |= GBLCTL_RCLKRST_ON;                                     // Clk
    while ( ( stMcAspRegs->u32RGBLCTL & GBLCTL_RCLKRST_ON ) != GBLCTL_RCLKRST_ON );

    stMcAspRegs->u32XSTAT = 0x0000ffff;        // Clear all
    stMcAspRegs->u32RSTAT = 0x0000ffff;        // Clear all

    stMcAspRegs->u32XGBLCTL |= GBLCTL_XSRCLR_ON;                                   // Serialize
    while ( ( stMcAspRegs->u32XGBLCTL & GBLCTL_XSRCLR_ON ) != GBLCTL_XSRCLR_ON );
    stMcAspRegs->u32RGBLCTL |= GBLCTL_RSRCLR_ON;                                      // Serialize
    while ( ( stMcAspRegs->u32RGBLCTL & GBLCTL_RSRCLR_ON ) != GBLCTL_RSRCLR_ON );

    /* Write a 0, so that no underrun occurs after releasing the state machine */
    stMcAspRegs->u32XBUF0 = 0;
    //MCASP2_RBUF0 = 0;

    stMcAspRegs->u32XGBLCTL |= GBLCTL_XSMRST_ON;                                       // State Machine
    while ( ( stMcAspRegs->u32XGBLCTL & GBLCTL_XSMRST_ON ) != GBLCTL_XSMRST_ON );
    stMcAspRegs->u32RGBLCTL |= GBLCTL_RSMRST_ON;                                       // State Machine
    while ( ( stMcAspRegs->u32RGBLCTL & GBLCTL_RSMRST_ON ) != GBLCTL_RSMRST_ON );

    stMcAspRegs->u32XGBLCTL |= GBLCTL_XFRST_ON;                                        // Frame Sync
    while ( ( stMcAspRegs->u32XGBLCTL & GBLCTL_XFRST_ON ) != GBLCTL_XFRST_ON );
    stMcAspRegs->u32RGBLCTL |= GBLCTL_RFRST_ON;                                        // Frame Sync
    while ( ( stMcAspRegs->u32RGBLCTL & GBLCTL_RFRST_ON ) != GBLCTL_RFRST_ON );

     /* Start by sending a dummy write */
    while( ! ( stMcAspRegs->u32SRCTL0 & 0x10 ) );  // Check for Tx ready
	stMcAspRegs->u32XBUF0 = 0;
    /* Play Tone */
    for ( u16Seconds = 0 ; u16Seconds < AIC3106_LO_TEST_DURATION ; u16Seconds++ )
    {
        for ( u16MilliSec = 0 ; u16MilliSec < 1000 ; u16MilliSec++ )
        {
            for ( u16Sample = 0 ; u16Sample < 48 ; u16Sample++ )
            {
                /* Send a sample to the left channel */
                while ( ! ( stMcAspRegs->u32SRCTL0 & 0x10 ) );
                    stMcAspRegs->u32XBUF0 = (sinetable[u16Sample] << 16);
                /* Send a sample to the right channel */
                while ( ! ( stMcAspRegs->u32SRCTL0 & 0x10 ) );
                    stMcAspRegs->u32XBUF0 = (sinetable[u16Sample] << 16);
			}
        }
    }

    /* Close Codec */ 
    aic3106RegSet( AIC3106_PAGESELECT, 0 ); // Select Page 0
    aic3106RegSet( AIC3106_RESET, 0x80 );   // Reset the AIC3106
    
    /* Close McASP */
    stMcAspRegs->u32SRCTL0 = 0; // Serializers
    stMcAspRegs->u32SRCTL1 = 0;
    stMcAspRegs->u32GBLCTL = 0;  // Global Reset
	
	return (u32RetVal);
}
